#!/usr/bin/Rscript

# load libraries for PostgreSQL and fast datafiles parsing
library(RPostgreSQL)
library(data.table)

## DATABASE CONNECTION PARAMETERS ##
host <- "localhost"
port <- 5432
user <- "USERNAME"
password <- "PASSWORD"
dbname <- "mapi_demo"
schema <- "public"

## FILES ##
file.samples <- "sim2_12_samples.csv" # sample file with sample name, x y coordinates and group if any
file.metrics <- "sim2_12.arp.gp.spa.out" # metrics file with sample names and metric value
file.metrics.columns <- c("name_i", "name_j", "i", "j", "sd", "pw", "all_loci") # spagedi header
file.results.pdf <- "sim2_12.pdf" # final map file

## TABLES & VIEWS NAMES ##
table.samples <- "t_sim2_12_samples"
table.metrics <- "t_sim2_12_metrics"
table.results <- "t_sim2_12_results"
view.samples <- paste("v_",table.samples, sep="")
view.metrics <- paste("v_",table.metrics, sep="")

# database connection
con <- dbConnect(PostgreSQL(), host=host, port=port, user=user, password=password, dbname=dbname)
# clean existing views ot tables, if any
dbSendQuery(con, paste("DROP VIEW IF EXISTS ",paste(schema, view.samples, sep="."),";"))
dbSendQuery(con, paste("DROP VIEW IF EXISTS ",paste(schema, view.metrics, sep="."),";", sep=""))
dbSendQuery(con, paste("DROP TABLE IF EXISTS ",paste(schema, table.samples, sep=".")," CASCADE;", sep=""))
dbSendQuery(con, paste("DROP TABLE IF EXISTS ",paste(schema, table.metrics, sep=".")," CASCADE;", sep=""))

# load samples
s <- as.data.frame(fread(file.samples, header=TRUE, sep=","))
dbWriteTable(con, c(schema, table.samples), s, overwrite=TRUE, row.names=FALSE)
# load metrics
d <- as.data.frame(fread(file.metrics, header=TRUE))
colnames(d) <- file.metrics.columns
dbWriteTable(con, c(schema, table.metrics), d, overwrite=TRUE, row.names=FALSE)

## VIEWS FORMATTED FOR MAPI ##
# view for samples, rename columns and computes geographical information
dbSendQuery(con, paste("CREATE VIEW ",paste(schema, view.samples, sep=".")," AS
	SELECT ind::text AS point_name, 
		sampling_year::text AS point_group, 
		ST_SetSRID(ST_MakePoint(x, y), 3258) AS point_geom
	FROM  ",paste(schema, table.samples, sep="."),";", sep=""))

# view for metrics, keep only mandatory columns
dbSendQuery(con, paste("CREATE VIEW ",paste(schema, view.metrics, sep=".")," AS 
		SELECT name_i AS point_1_name, name_j AS point_2_name, all_loci AS relation_value
		FROM  ",paste(schema, table.metrics, sep="."),";", sep=""))

## MAPI PARAMETERS ##
mapi.beta <- 0.5 # 0.5 for regular sampling, 0.25 for random sampling
mapi.nb_permutations <- 1000 # 1000 ok for 5%
mapi.eccentricity <- 0.975 # 0.975 is default value
mapi.error_circle <- 10 # 10 m is typical GPS error
mapi.inter_groups <- "false" # not relevant here (no groups)
mapi.min_distance <- "NULL" # "NULL" for no distance filtering
mapi.max_distance <- "NULL" # "NULL" for no distance filtering
mapi.alpha <- 0.05 # 0.05 for 5% tails

## START AUTOMATIC MAPI COMPUTATION
n <- dbGetQuery(con, paste("SELECT MAPI_RunAuto(
			 '",view.samples,"', 
			 '",view.metrics,"', 
			 ",mapi.beta,",
			 '",table.results,"', 
			 '",schema,"', 
			 ",mapi.eccentricity,", 
			 ",mapi.error_circle,", 
			 ",mapi.nb_permutations,", 
			 ",mapi.inter_groups,", 
			 ",mapi.min_distance,", 
			 ",mapi.max_distance,",
			 ",mapi.alpha,");", sep=""), verbose=TRUE)
# ... be patient (~ half-hour)

## MAPPING WITH R (LINUX)
# load libraries for mapping
library(sp)
library(rgdal)
library(RColorBrewer)
library(lattice)
library(latticeExtra)

file.results.pdf <- "sim2_12_lin.pdf" # final map file

# reads spatial tables
OGRstring <- paste("PG:host=",host," port=",port," user=",user," password=",password," dbname=",dbname, sep="")
if (schema == "public") { # surprisingly, OGR don't accept "public" as a schema...
	samp <- readOGR(OGRstring, view.samples)
	resu <- readOGR(OGRstring, table.results)
	table.tails <- paste(table.results,"tails", sep="_")
} else {
	samp <- readOGR(OGRstring, paste(schema, view.samples, sep="."))
	resu <- readOGR(OGRstring, paste(schema, table.results, sep="."))
	table.tails <- paste(schema, paste(table.results,"tails", sep="_"), sep=".")
}
tails.info <- ogrInfo(OGRstring, table.tails)
if (tails.info$nrows > 0) { # read tails table only if data available
	tails <- readOGR(OGRstring, table.tails)
	# filter "BY" method and separates lower and upper tails
	lower.tail <- tails[tails$tail=="lower" & tails$method=="M_BY", ]
	upper.tail <- tails[tails$tail=="upper" & tails$method=="M_BY", ]
} else { # nothing to plot, but empty new object avoids crash
	lower.tail <- new("SpatialPolygonsDataFrame")
	upper.tail <- new("SpatialPolygonsDataFrame")
}

# colorscale
aec <- rainbow(512, start=0, end=0.8)
vals <- sort(resu$averaged_value)
scale.min <- vals[1]
scale.max <- vals[length(vals)]
scale.by <- (scale.max - scale.min) / 20
at.std <- seq(scale.min, scale.max+scale.by, by=scale.by)
at <- sort(at.std)

# spatial plot
pl <- spplot(resu, "averaged_value", col="transparent", col.regions=aec, colorkey=TRUE, at=at, 
		main=list(label=table.results), 
		sp.layout=c("sp.points", samp, col="black", pch=15, cex=0.7))
if (nrow(upper.tail@data) > 0) { pl <- pl + layer(sp.polygons(upper.tail, fill=c("transparent"), lwd=5 )) }
if (nrow(lower.tail@data) > 0) { pl <- pl + layer(sp.polygons(lower.tail, fill=c("transparent"), lwd=2 )) }
# print plot to pdf
pdf(file.results.pdf)
print(pl) # print !!!
dev.off()

# disconnect
dbDisconnect(con)
