% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MAPI_RunOnGrid.R
\name{MAPI_RunOnGrid}
\alias{MAPI_RunOnGrid}
\title{Function MAPI_RunOnGrid}
\usage{
MAPI_RunOnGrid(
  samples,
  metric,
  grid,
  isMatrix = FALSE,
  ecc = 0.975,
  errRad = 10,
  nbPermuts = 0,
  dMin = 0,
  dMax = Inf,
  use_s2 = FALSE,
  nbCores = parallel::detectCores() - 1,
  N = 8
)
}
\arguments{
\item{samples}{a data.frame with names and geographical coordinates of samples.
Column names must be: 'ind', 'x', 'y'.  Optional column 'errRad' with an error radius
for sample locations (eg. GPS uncertainty).
Coordinates should be projected (not latitude/longitude) for a local dataset, angular coordinates
(ie. latitude/longitude) are accepted for worldwide datasets.
In this case, the longitude will be in the "x" column and the latitude in the "y" column.}

\item{metric}{a data.frame or a square matrix with the pairwise metric computed for all pairs of samples.
If data.frame, column names must be: 'ind1', 'ind2', 'value'.
If matrix, sample names must be the row- and column names.}

\item{grid}{a spatial object of class 'sf' with the geometry of each cell.
When using your own grid, please check that the object structure is the same as returned by
\code{\link{MAPI_GridAuto}} or \code{\link{MAPI_GridHexagonal}}.}

\item{isMatrix}{Boolean. Depends on the 'metric' data:\cr
TRUE if 'metric' is a square matrix with column names = row names and standing for sample names.\cr
FALSE if 'metric is a three columns data.frame ('ind1', 'ind2', 'value'). \cr
The default value is determined using a "matrix" class detection for metric as well as identity between row and column number.}

\item{ecc}{ellipse eccentricity value (0.975 by default).}

\item{errRad}{global error radius for sample locations (same radius for all samples, 10 by default).
Units are in the same reference system as the sample geographical coordinates.
To use different error radius values for sample locations, add a column 'errRad' in the 'sample' data (see \code{\link{mapi}}).}

\item{nbPermuts}{number of permutations of sample locations (0 by default).}

\item{dMin}{minimum distance between individuals. 0 by default.}

\item{dMax}{maximal distance between individuals. +Inf by default.}

\item{use_s2}{transform grid and coordinates in latitude,longitude in order to use s2 library functions (default FALSE).
When set to TRUE MAPI is able to process worldwide grids and ellipses and generate results on the sphere.}

\item{nbCores}{number of CPU cores you want to use during parallel computation.
The default value is estimated as the number of available cores minus 1, suitable for a personal computer.
On a cluster you might have to set it to a reasonable value (eg. 8) in order to keep resources for other tasks.}

\item{N}{number of points used per quarter of ellipse, 8 by default.
Don't change it unless you really know what you are doing.}
}
\value{
a spatial object of class 'sf' providing for each cell: \cr
\itemize{
\item gid: Cell ID \cr
\item x and y coordinates of cell center \cr
\item nb_ell: number of ellipses used to compute the weighted mean \cr
\item avg_value: weighted mean of the pairwise metric \cr
\item sum_wgts: sum of weights of ellipses used to compute the weighted mean \cr
\item w_stdev: weighted standard deviation of the pairwise metric \cr
\item swQ: percentile of the sum of weights \cr
\item geometry \cr
When permutations are performed: \cr
\item proba: proportion of the permuted weighted means below the observed weighted mean \cr
\item ltP: lower-tail p-value adjusted using the FDR procedure of Benjamini and Yekutieli \cr
\item utP: upper-tail p-value adjusted using the FDR procedure of Benjamini and Yekutieli \cr
}
}
\description{
Launch a MAPI analysis for a given grid computed with \code{\link{MAPI_GridAuto}}
or \code{\link{MAPI_GridHexagonal}} or provided by users.
}
\details{
To test whether the pairwise metric values associated with the ellipses are independent of the sample locations, those are permuted 'nbPermuts' times.
At each permutation, new cell values are computed and stored to build a cumulative null distribution for each cell of the grid.
Each cell value from the observed data set is then ranked against its null distribution.
For each cell, the proportion of permuted values that are smaller or greater
than the observed value provides a lower-tailed (ltP) and upper-tailed (utP) test p-value.

A false discovery rate (FDR) procedure (Benjamini and Yekutieli, 2001) is applied to account for multiple
testing (number of cells) under positive dependency conditions (spatial autocorrelation).  An adjusted
p-value is computed for each cell using the function \code{p.adjust} from the 'stats' package with the method 'BY'.
}
\examples{
\dontrun{
data(metric)
data(samples)
my.grid <- MAPI_GridHexagonal(samples, crs=3857, 500) # 500m halfwidth

# Note: 10 permutations is only for test purpose, increase to >=1000 in real life!
my.results <- MAPI_RunOnGrid(samples, metric, grid=my.grid, nbPermuts=10, nbCores=1)

# eg. Export results to shapefile "myFirstMapiResult" in current directory
library(sf)
st_write(my.results, dsn=".", layer="myFirstMapiResult", driver="ESRI Shapefile")
}

}
\references{
Benjamini, Y. and Yekutieli, D. (2001). The control of the false discovery rate in multiple testing under dependency. Annals of Statistics 29, 1165–1188.
}
